
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; MODULE      : math-speech-fr.scm
;; DESCRIPTION : mathematical editing using French speech
;; COPYRIGHT   : (C) 2022  Joris van der Hoeven
;;
;; This software falls under the GNU general public license version 3 or later.
;; It comes WITHOUT ANY WARRANTY WHATSOEVER. For details, see the file LICENSE
;; in the root directory or <http://www.gnu.org/licenses/gpl-3.0.html>.
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(texmacs-module (math math-speech-fr)
  (:use (math math-speech)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Sanitize input
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define-table french-numbers
  ("0" "zéro") ("1" "un") ("2" "deux") ("3" "trois") ("4" "quatre")
  ("5" "cinq") ("6" "six") ("7" "sept") ("8" "huit") ("9" "neuf"))

(define-table french-ambiguate
  ("deux" "2/de"))

(define (string-table-replace s t)
  (with repl (lambda (x) (with y (ahash-ref t x) (if y (car y) x)))
    (with l (string-decompose s " ")
      (string-recompose (map repl l) " "))))

(tm-define (speech-pre-sanitize lan s)
  (:require (== lan 'french))
  (set! s (spaced-quotes s))
  s)

(define (rewrite-/ s)
  (with l (string-decompose s "/")
    (if (and (== (length l) 2)
             (string-number? (car l))
             (string-number? (cadr l)))
        (string-replace s "/" " @sur ")
        s)))

(tm-define (speech-sanitize lan mode s)
  (:require (and (== lan 'french) (== mode 'math)))
  (set! s (locase-all s))
  (set! s (letterize s))
  (set! s (list->tmstring (clean-letter-digit (tmstring->list s))))
  (set! s (string-recompose (map rewrite-/ (string-decompose s " ")) " "))
  (set! s (spaced-quotes s))
  (set! s (string-replace s "+" " plus "))
  (set! s (string-replace-trailing s "-" " moins "))
  (set! s (string-replace s "<times>" " fois "))
  (set! s (string-replace-trailing s "." " point "))
  (set! s (string-replace s "=" " égal "))
  (set! s (string-replace s "," " virgule "))
  (set! s (string-replace s ":" " double points "))
  (set! s (string-replace s ";" " point virgule "))
  (set! s (string-replace s "^" " chapeau "))
  (set! s (string-replace s "~" " tilde "))
  (set! s (string-replace s "(" " ( "))
  (set! s (string-replace s ")" " ) "))
  (set! s (string-replace s "[" " [ "))
  (set! s (string-replace s "]" " ] "))
  (set! s (string-replace s "{" " { "))
  (set! s (string-replace s "}" " } "))
  (set! s (string-replace s "<ldots>" " points "))
  (set! s (string-replace s "<cdots>" " points "))
  (set! s (string-table-replace s french-numbers))
  (set! s (string-table-replace s french-ambiguate))
  (set! s (string-replace s "  " " "))
  (set! s (string-replace s "  " " "))
  (set! s (tm-string-trim-both s))
  (set! s (french-normalize 'math s))
  s)

(speech-collection dont-break french
  "ah" "ai" "an" "ar" "as" "at" "au"
  "el" "en" "es" "et" "eu" "ex" "ét"
  "id" "il"
  "oh" "ok" "on" "os" "ou"
  "un"
  "ye" "yo"

  "la" "ma" "ta" "za"
  "ce" "de" "he" "je" "le" "ne" "se" "te"
  "bi" "hi" "ji" "pi" "si" "ti" "xi"
  "ho" "no" "to"
  "du" "mu" "nu" "tu"
  "sy")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Normalization of French text (singular/plural, masculin/feminin, etc.)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define (french-resuffix* mode s suf1 suf2)
  (and (string-ends? s suf1)
       (let* ((l1 (string-length s))
              (l2 (string-length suf1))
              (r  (string-append (substring s 0 (- l1 l2)) suf2)))
         (and (>= l1 (+ l2 3))
              (raw-speech-accepts? 'french mode r)
              r))))

(define (french-resuffix mode s suf1 suf2)
  (or (french-resuffix* mode s suf1 suf2)
      (french-resuffix* mode s suf2 suf1)))

(define (french-normalize-one mode s)
  (or (and (raw-speech-accepts? 'french mode s) s)
      (french-resuffix mode s "e" "")
      (french-resuffix mode s "s" "")
      (french-resuffix mode s "es" "")
      (french-resuffix mode s "t" "")
      (french-resuffix mode s "x" "")
      (french-resuffix mode s "aux" "al")
      (french-resuffix mode s "ère" "er")
      (french-resuffix mode s "ères" "ers")
      (french-resuffix mode s "ères" "er")
      (french-resuffix mode s "ve" "f")
      (french-resuffix mode s "ves" "f")
      (french-resuffix mode s "ves" "fs")
      (french-resuffix mode s "er" "")
      (french-resuffix mode s "er" "e")
      (french-resuffix mode s "er" "es")
      (french-resuffix mode s "er" "ent")
      (french-resuffix mode s "er" "é")
      (french-resuffix mode s "er" "ée")
      (french-resuffix mode s "er" "és")
      (french-resuffix mode s "er" "ées")
      (french-resuffix mode s "ir" "")
      (french-resuffix mode s "ir" "e")
      (french-resuffix mode s "ir" "es")
      (french-resuffix mode s "ir" "ent")
      (french-resuffix mode s "tir" "s")
      (french-resuffix mode s "re" "")
      (french-resuffix mode s "re" "s")
      (french-resuffix mode s "re" "ent")
      (french-resuffix mode s "qu'" "que")
      s))

(define (french-normalize-compute mode s)
  (let* ((l (string-decompose s " "))
         (r (map (cut french-normalize-one mode <>) l)))
    (string-recompose r " ")))

(define french-normal-table (make-ahash-table))

(tm-define (french-normalize mode s)
  (with key (list mode s)
    (when (not (ahash-ref french-normal-table key))
      (ahash-set! french-normal-table key (french-normalize-compute mode s)))
    (ahash-ref french-normal-table key)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Tables for recognizing mathematics inside text
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(speech-collection prefix french
  "grand" "petit")

(speech-collection postfix french
  "majuscule" "minuscule"
  "gras" "droit" "calligraphique" "fraktur" "gothique"
  "tableau noir gras" "sans serif" "machine à écrire")

(speech-collection prefix french
  "exponentielle" "logarithme" "sinus" "cosinus" "tangente"
  "racine carrée")

(speech-collection postfix french
  "prime" "factoriel" "carré" "cube")

(speech-collection math-mode french
  "math" "maths" "matt" "mathématiques")

(speech-collection text-mode french
  "text" "texte")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Entering mathematical symbols via French speech
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(speech-symbols french
  ("zéro" "0")
  ("un" "1")
  ("deux" "2")
  ("trois" "3")
  ("quatre" "4")
  ("cinq" "5")
  ("six" "6")
  ("sept" "7")
  ("huit" "8")
  ("neuf" "9")
  ("dix" "10")
  ("cent" "100")
  ("mille" "1000")
  ("million" "1000000")
  ("milliard" "1000000000")

  ("a" "a")
  ("à" "a")
  ("b" "b")
  ("c" "c")
  ("d" "d")
  ("e" "e")
  ("f" "f")
  ("g" "g")
  ("h" "h")
  ("i" "i")
  ("j" "j")
  ("k" "k")
  ("l" "l")
  ("m" "m")
  ("n" "n")
  ("o" "o")
  ("p" "p")
  ("q" "q")
  ("r" "r")
  ("s" "s")
  ("t" "t")
  ("u" "u")
  ("v" "v")
  ("w" "w")
  ("x" "x")
  ("y" "y")
  ("z" "z")

  ("alpha" "<alpha>")
  ("beta" "<beta>")
  ("gamma" "<gamma>")
  ("delta" "<delta>")
  ("epsilon" "<epsilon>")
  ("zeta" "<zeta>")
  ("eta" "<eta>")
  ("theta" "<theta>")
  ("iota" "<iota>")
  ("kappa" "<kappa>")
  ("lambda" "<lambda>")
  ("mu" "<mu>")
  ("nu" "<nu>")
  ("xi" "<xi>")
  ("omicron" "<omicron>")
  ("pi" "<pi>")
  ("rho" "<rho>")
  ("sigma" "<sigma>")
  ("tau" "<tau>")
  ("upsilon" "<upsilon>")
  ("phi" "<phi>")
  ("psi" "<psi>")
  ("chi" "<chi>")
  ("omega" "<omega>")

  ("constante e" "<mathe>")
  ("constante i" "<mathi>")
  ("constante pi" "<mathpi>")
  ("constante gamma" "<mathgamma>")
  ("constante d'euler" "<mathgamma>")
  ("nombre e" "<mathe>")
  ("nombre i" "<mathi>")
  ("nombre pi" "<mathpi>")
  ("nombre gamma" "<mathgamma>")

  ("infini" "<infty>")
  ("complexes" "<bbb-C>")
  ("entiers positifs" "<bbb-N>")
  ("rationnels" "<bbb-Q>")
  ("réels" "<bbb-R>")
  ("entiers" "<bbb-Z>")

  ("plus" "+")
  ("moins" "-")
  ("fois" "*")
  ("croix" "<times>")
  ("appliquer" " ")
  ("espace" " ")
  ("rond" "<circ>")
  ("tenseur" "<otimes>")
  ("factoriel" "!")

  ("égal" "=")
  ("non égal" "<neq>")
  ("défini par" "<assign>")
  ("congru" "<equiv>")
  ("inférieur" "<less>")
  ("inférieur égal" "<leqslant>")
  ("supérieur" "<gtr>")
  ("supérieur égal" "<geqslant>")
  ("dans" "<in>")
  ("pas dans" "<nin>")
  ("contient" "<ni>")
  ("contient pas" "<nni>")
  ("inclus dans" "<subset>")
  ("strictement inclus dans" "<subset>")
  ("inclus dans égal" "<subseteq>")
  ("inclut" "<supset>")
  ("inclut strictement" "<supset>")
  ("inclut égal" "<supseteq>")

  ("similaire" "<sim>")
  ("asymptotique" "<asymp>")
  ;;("approx" "<approx>")
  ("isomorphe" "<cong>")
  ("négligeable" "<prec>")
  ("dominé" "<preccurlyeq>")
  ("domine" "<succcurlyeq>")
  ("domine strictement" "<succ>")

  ("flèche" "<rightarrow>")
  ("longue flèche" "<rightarrow>")

  ("pour tout" "<forall>")
  ("existe" "<exists>")
  ("ou" "<vee>")
  ("et" "<wedge>")
  ("implique" "<Rightarrow>")
  ("équivaut" "<Leftrightarrow>")

  ("point" ".")
  ("virgule" ",")
  ("double point" ":")
  ("point virgule" ";")
  ("point d'exclamation" "!")
  ("point d'interrogation" "?")
  ("." ".")
  ("," ",")
  (":" ":")
  (";" ";")
  ("!" "!")
  ("?" "?")
  ("tel que" "<suchthat>")
  )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; More complex mathematical speech commands
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(speech-map french math
  ("grand" (speech-alter-letter :big))
  ("petit" (speech-alter-letter :small))
  ("majuscule" (speech-alter-letter* :big))
  ("minuscule" (speech-alter-letter* :small))
  ("gras" (speech-alter-letter* :bold))
  ("medium" (speech-alter-letter* :medium))
  ("droit" (speech-alter-letter* :up))
  ("italique" (speech-alter-letter* :it))
  ("calligraphique" (speech-alter-letter* :cal))
  ("fraktur" (speech-alter-letter* :frak))
  ("tableau noir gras" (speech-alter-letter* :bbb))
  ("double" (speech-alter-letter* :bbb))
  ("normal" (speech-alter-letter* :normal))
  ("sans serif" (speech-alter-letter* :ss))
  ("machine à écrire" (speech-alter-letter* :tt))
  ("opérateur" (speech-operator))

  ("facteur" (speech-factor))
  ("inverse" (speech-insert-superscript "-1"))
  ("carré" (speech-insert-superscript "2"))
  ("cube" (speech-insert-superscript "3"))
  ("transpose" (speech-insert-superscript "<top>"))
  ("indice" (speech-short-subscript))
  ("grand indice" (speech-subscript))
  ("puissance" (speech-short-superscript))
  ("exposant" (speech-superscript))
  ("début indice" (make 'rsub))
  ("début exposant" (make 'rsup))
  ("fin indice" (speech-end 'rsub))
  ("fin exposant" (speech-end 'rsup))

  ("prime" (make-rprime "'"))
  ("double prime" (make-rprime "'") (make-rprime "'"))
  ("triple prime" (make-rprime "'") (make-rprime "'") (make-rprime "'"))
  ("adjoint" (make-rprime "<asterisk>"))

  ("chapeau" (speech-accent "^"))
  ("tilde" (speech-accent "~"))
  ("barre" (speech-accent "<bar>"))
  ("large chapeau" (speech-wide "^"))
  ("large tilde" (speech-wide "~"))
  ("large barre" (speech-wide "<bar>"))
  ("dessous" (speech-under))

  ("de" (speech-of))
  ("ouvrir" (speech-open "(" ")"))
  ("fermer" (speech-close))
  ("parenthèses" (speech-brackets "(" ")"))
  ("crochets" (speech-brackets "[" "]"))
  ("accolades" (speech-brackets "{" "}"))
  ("chevrons" (speech-brackets "<langle>" "<rangle>"))
  ("partie entière" (speech-brackets "<lfloor>" "<rfloor>"))
  ("ouvrir parenthèses" (speech-open "(" ")"))
  ("ouvrir crochets" (speech-open "[" "]"))
  ("ouvrir accolades" (speech-open "{" "}"))
  ("ouvrir chevrons" (speech-open "<langle>" "<rangle>"))
  ("ouvrir partie entière" (speech-open "<lfloor>" "<rfloor>"))

  ("(" (speech-open "(" ")"))
  ("[" (speech-open "[" "]"))
  ("{" (speech-open "{" "}"))
  (")" (speech-close))
  ("]" (speech-close))
  ("}" (speech-close))
  
  ("arc cos" (speech-insert-operator "arccos"))
  ("arc sin" (speech-insert-operator "arcsin"))
  ("arc tan" (speech-insert-operator "arctan"))
  ("arg" (speech-insert-operator "arg"))
  ("cos" (speech-insert-operator "cos"))
  ("deg" (speech-insert-operator "deg"))
  ("det" (speech-insert-operator "det"))
  ("dim" (speech-insert-operator "dim"))
  ("exp" (speech-insert-operator "exp"))
  ("gcd" (speech-insert-operator "gcd"))
  ("log" (speech-insert-operator "log"))
  ("hom" (speech-insert-operator "hom"))
  ("inf" (speech-insert-operator "inf"))
  ("ker" (speech-insert-operator "ker"))
  ("lcm" (speech-insert-operator "lcm"))
  ("lim" (speech-insert-operator "lim"))
  ("lim inf" (speech-insert-operator "liminf"))
  ("lim sup" (speech-insert-operator "limsup"))
  ("ln" (speech-insert-operator "ln"))
  ("log" (speech-insert-operator "log"))
  ("max" (speech-insert-operator "max"))
  ("min" (speech-insert-operator "min"))
  ("Pr" (speech-insert-operator "Pr"))
  ("sin" (speech-insert-operator "sin"))
  ("supremum" (speech-insert-operator "sup"))
  ("tan" (speech-insert-operator "tan"))

  ("mod" (speech-insert-infix-operator "mod"))
  ("div" (speech-insert-infix-operator "div"))
  ("quo" (speech-insert-infix-operator "quo"))
  ("rem" (speech-insert-infix-operator "rem"))
  ("division" (speech-insert-infix-operator "div"))
  ("modulo" (speech-insert-infix-operator "mod"))
  ("quotient" (speech-insert-infix-operator "quo"))
  ("reste" (speech-insert-infix-operator "rem"))
  ("pseudo reste" (speech-insert-infix-operator "prem"))

  ("plus points plus" (speech-dots "+" "<cdots>"))
  ("moins points moins" (speech-dots "-" "<cdots>"))
  ("fois points fois" (speech-dots "*" "<cdots>"))
  ("virgule points virgule" (speech-dots "," "<ldots>"))
  ("et points et" (speech-dots "<wedge>" "<cdots>"))
  ("ou points ou" (speech-dots "<vee>" "<cdots>"))
  ("égal points égal" (speech-dots "=" "<cdots>"))
  ("similaire points similaire" (speech-dots "<sim>" "<cdots>"))
  ("inférieur points inférieur" (speech-dots "<less>" "<cdots>"))
  ("inférieur égal points inférieur égal" (speech-dots "<leqslant>" "<cdots>"))
  ("supérieur points supérieur" (speech-dots "<gtr>" "<cdots>"))
  ("supérieur égal points supérieur égal" (speech-dots "<geqslant>" "<cdots>"))
  ("rond points rond" (speech-dots "<circ>" "<cdots>"))
  ("tenseur points tenseur" (speech-dots "<otimes>" "<cdots>"))

  ("somme" (speech-big-operator "sum"))
  ("produit" (speech-big-operator "prod"))
  ("intégrale" (speech-big-operator "int"))
  ("intégrale contours" (speech-big-operator "oint"))
  ("intégrale double" (speech-big-operator "iint"))
  ("intégrale triple" (speech-big-operator "iiint"))
  ("double intégrale" (speech-big-operator "iint"))
  ("triple intégrale" (speech-big-operator "iiint"))
  ("pour" (speech-for))
  ("jusqu'à" (speech-until))

  ("racine carrée" (speech-sqrt-of))
  ("racine carrée de" (speech-sqrt-of))
  ("racine carrée 2/de" (speech-sqrt-of))
  ("début racine carrée" (speech-sqrt))
  ("fin racine carrée" (speech-end 'sqrt))
  ("sur" (speech-over))
  ("@sur" (speech-short-over))
  ("fraction" (speech-fraction))
  ("début fraction" (speech-fraction))
  ("numérateur" (go-to-fraction :numerator))
  ("dénominateur" (go-to-fraction :denominator))
  ("fin fraction" (speech-end 'frac))

  ("matrice" (make 'matrix))
  ("déterminant" (make 'det))
  ("choix" (make 'choice))
  ("points horizontaux" (insert "<cdots>"))
  ("points verticaux" (insert "<vdots>"))
  ("points diagonaux" (insert "<ddots>"))
  ("points montants" (insert "<udots>"))

  ;;("more" "var")
  )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Commonly used unambiguous words for letters
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(speech-reduce french math
  ("alice" "a")
  ("anatole" "a")
  ("anna" "a")
  ("arthur" "a")
  ("bernard" "b")
  ("berthe" "b")
  ("bruxelles" "b")
  ("cécile" "c")
  ("célestine" "c")
  ("césar" "c")
  ("charles" "c")
  ("daniel" "d")
  ("david" "d")
  ("désiré" "d")
  ("édouard" "e")
  ("émile" "e")
  ("eugène" "e")
  ("françois" "f")
  ("frédéric" "f")
  ("gaston" "g")
  ("george" "g")
  ("gustave" "g")
  ("henri" "h")
  ("henry" "h")
  ("ida" "i")
  ("irma" "i")
  ("isabelle" "i")
  ("isidor" "i")
  ("isidore" "i")
  ("jacques" "j")
  ("jeanne" "j")
  ("joseph" "j")
  ("juventus" "j")
  ("kilo" "k")
  ("kilogramme" "k")
  ("kléber" "k")
  ("leopold" "l")
  ("louis" "l")
  ("louise" "l")
  ("marcel" "m")
  ("marie" "m")
  ("nicole" "n")
  ("nicolas" "n")
  ("napoléon" "n")
  ("olga" "o")
  ("oscar" "o")
  ("paul" "p")
  ("piano" "p")
  ("pierre" "p")
  ("québec" "q")
  ("quiévrain" "q")
  ("quintal" "q")
  ("quittance" "q")
  ("raoul" "r")
  ("robert" "r")
  ("samuel" "s")
  ("simon" "s")
  ("suzanne" "s")
  ("téléphone" "t")
  ("thérèse" "t")
  ("thomas" "t")
  ("ulysse" "u")
  ("ursule" "u")
  ("victor" "v")
  ("waterloo" "w")
  ("william" "w")
  ("xantippe" "x")
  ("xavier" "x")
  ("yvonne" "y")
  ("zoé" "z")
  ("zurich" "z"))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Speech reductions
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(speech-reduce french math
  ("lettre a" "a")
  ("lettre b" "b")
  ("lettre c" "c")
  ("lettre d" "d")
  ("lettre e" "e")
  ("lettre f" "f")
  ("lettre g" "g")
  ("lettre h" "h")
  ("lettre i" "i")
  ("lettre j" "j")
  ("lettre k" "k")
  ("lettre l" "l")
  ("lettre m" "m")
  ("lettre n" "n")
  ("lettre o" "o")
  ("lettre p" "p")
  ("lettre q" "q")
  ("lettre r" "r")
  ("lettre s" "s")
  ("lettre t" "t")
  ("lettre u" "u")
  ("lettre v" "v")
  ("lettre w" "w")
  ("lettre x" "x")
  ("lettre y" "y")
  ("lettre z" "z")

  ("le nombre" "nombre")
  ("la constante" "constante")
  ("la lettre" "lettre")

  ("les complexes" "complexes")
  ("les entiers positifs" "entiers positifs")
  ("les rationnels" "rationnels")
  ("les réels" "réels")
  ("les entiers" "entiers")
  ("nombres complexes" "complexes")
  ("nombres entiers positifs" "entiers positifs")
  ("nombres rationnels" "rationnels")
  ("nombres réels" "réels")
  ("nombres entiers" "entiers")

  ("partie entière de" "partie entière")
  ("partie entière 2/de" "partie entière")
  ("fermer parenthèses" "fermer")
  ("fermer crochets" "fermer")
  ("fermer accolades" "fermer")
  ("fermer chevrons" "fermer")
  ("fermer partie entière" "fermer")

  ("ouvrir le" "ouvrir")
  ("ouvrir la" "ouvrir")
  ("ouvrir les" "ouvrir")
  ("ouvrir un" "ouvrir")
  ("ouvrir une" "ouvrir")
  ("ouvrir des" "ouvrir")
  ("fermer le" "fermer")
  ("fermer la" "fermer")
  ("fermer les" "fermer")
  ("fermer un" "fermer")
  ("fermer une" "fermer")
  ("fermer des" "fermer")

  ("ensemble" "accolades")
  ("ensemble de" "accolades")
  ("ensemble 2/de" "accolades")
  ("ensemble des" "accolades")
  ("l'ensemble" "accolades")
  ("l'ensemble de" "accolades")
  ("l'ensemble 2/de" "accolades")
  ("l'ensemble des" "accolades")

  ("débuter" "début")
  ("démarrer" "commencer")
  ("terminer" "finir")
  ("un indice" "indice")
  ("l'indice" "indice")
  ("de l'indice" "indice")
  ("un exposant" "exposant")
  ("l'exposant" "exposant")
  ("de l'exposant" "exposant")
  ("une racine carrée" "racine carrée")
  ("la racine carrée" "racine carrée")
  ("de la racine carrée" "racine carrée")
  ("une fraction" "fraction")
  ("la fraction" "fraction")
  ("de la fraction" "fraction")
  ("commencer indice" "début indice")
  ("commencer exposant" "début exposant")
  ("commencer racine carrée" "début racine carrée")
  ("commencer fraction" "début fraction")
  ("insérer indice" "début indice")
  ("insérer exposant" "début exposant")
  ("insérer racine carrée" "début racine carrée")
  ("insérer fraction" "début fraction")
  ("finir indice" "fin indice")
  ("finir exposant" "fin exposant")
  ("finir racine carrée" "fin racine carrée")
  ("finir fraction" "fin fraction")
  ("sortir indice" "fin indice")
  ("sortir exposant" "fin exposant")
  ("sortir racine carrée" "fin racine carrée")
  ("sortir fraction" "fin fraction")
  ("sortir de l'indice" "fin indice")
  ("sortir de l'exposant" "fin exposant")
  ("sortir de l' indice" "fin indice")
  ("sortir de l' exposant" "fin exposant")
  ("sortir de la racine carrée" "fin racine carrée")
  ("sortir de la fraction" "fin fraction")
  ("transposé" "transpose")

  ("égal à" "égal")
  ("est égal à" "égal")

  ("pas égal" "non égal")
  ("pas égal à" "non égal")
  ("n'est pas égal à" "non égal")
  ("différent" "non égal")
  ("différent de" "non égal")
  ("différent 2/de" "non égal")
  ("est différent de" "non égal")
  ("est différent 2/de" "non égal")

  ("congru à" "congru")
  ("est congru à" "congru")
  
  ("plus petit" "inférieur")
  ("plus petit que" "inférieur")
  ("est plus petit" "inférieur")
  ("est plus petit que" "inférieur")
  ("est inférieur" "inférieur")
  ("inférieur à" "inférieur")
  ("inférieur ou égal" "inférieur égal")

  ("plus grand" "supérieur")
  ("plus grand que" "supérieur")
  ("est plus grand" "supérieur")
  ("est plus grand que" "supérieur")
  ("est supérieur" "supérieur")
  ("supérieur à" "supérieur")
  ("supérieur ou égal" "supérieur égal")

  ("est dans" "dans")
  ("n'est pas dans" "pas dans")
  ("ne contient pas" "contient pas")
  ("est inclus dans" "inclus dans")
  ("est inclus ou égal à" "inclus dans égal")
  ("est inclus dans ou égal à" "inclus dans égal")
  ("est strictement inclus dans" "strictement inclus dans")
  ("proprement inclus dans" "strictement inclus dans")
  ("inclut proprement" "inclut strictement")
  ("partie de" "inclus dans")
  ("est une partie de" "inclus dans")
  ("sous ensemble" "inclus dans")
  ("sous-ensemble" "sous ensemble")
  ("un sous ensemble" "sous ensemble")
  ("sous ensemble de" "inclus dans")
  ("sous ensemble 2/de" "inclus dans")
  ("sur ensemble" "inclut")
  ("sur-ensemble" "sur ensemble")
  ("un sur ensemble" "sur ensemble")
  ("sur ensemble de" "inclut")
  ("sur ensemble 2/de" "inclut")
  ("vers" "flèche")

  ("il existe" "existe")
  ("existe un" "existe")
  ("si seulement si" "équivaut")
  ("si et seulement si" "équivaut")

  ("appliquer à" "appliquer")
  ("la racine carrée" "racine carrée")

  ("arc cosinus" "arc cos")
  ("arc sinus" "arc sin")
  ("arc tangente" "arc tan")
  ("arccosinus" "arc cos")
  ("arcsinus" "arc sin")
  ("arctangente" "arc tan")
  ("argument" "arg")
  ("cosinus" "cos")
  ("degré" "deg")
  ("dimension" "dim")
  ("exponentielle" "exp")
  ("noyau" "ker")
  ("limite" "lim")
  ("limite inférieure" "lim inf")
  ("limite supérieure" "lim sup")
  ("lim inférieure" "lim inf")
  ("lim supérieure" "lim sup")
  ("logarithme naturel" "ln")
  ("logarithme naturels" "ln")
  ("logarithme népérien" "ln")
  ("logarithme" "log")
  ("maximum" "max")
  ("minimum" "min")
  ("probabilité" "Pr")
  ("sinus" "sin")
  ("tangente" "tan")

  ("l'arg" "arg")
  ("l'argument" "arg")
  ("le cos" "cos")
  ("le deg" "deg")
  ("le det" "det")
  ("la dim" "dim")
  ("la dimension" "dim")
  ("l'exp" "exp")
  ("l'exponentielle" "exp")
  ("le gcd" "gcd")
  ("le ker" "ker")
  ("le lcm" "lcm")
  ("la lim" "lim")
  ("la lim inf" "lim inf")
  ("la lim sup" "lim sup")
  ("la limite" "limite")
  ("la limite inférieure" "lim inf")
  ("la limite supérieure" "lim sup")
  ("le ln" "ln")
  ("le log" "log")
  ("le max" "max")
  ("le min" "min")
  ("la Pr" "Pr")
  ("la pr" "Pr")
  ("la probabilité" "Pr")
  ("le sin" "sin")
  ("la tan" "tan")
  ("la tangente" "tan")

  ("etc." "points")
  ("petit point" "points")
  ("petits points" "points")
  ("trois petits points" "points")
  ("point point point" "points")
  ("plus plus" "plus points plus")
  ("fois fois" "fois points fois")
  ("virgule virgule" "virgule points virgule")
  ("rond rond" "rond points rond")
  ("tenseur tenseur" "tenseur points tenseur")
  ("plus jusqu'à" "plus points plus")
  ("fois jusqu'à" "fois points fois")
  ("virgule jusqu'à" "virgule points virgule")
  ("et jusqu'à" "et points et")
  ("ou jusqu'à" "ou points ou")
  ("égal jusqu'à" "égal points égal")
  ("similaire jusqu'à" "similaire points similaire")
  ("inférieur jusqu'à" "inférieur points inférieur")
  ("inférieur égal jusqu'à" "inférieur égal points inférieur égal")
  ("supérieur jusqu'à" "supérieur points supérieur")
  ("supérieur égal jusqu'à" "supérieur égal points supérieur égal")
  ("rond jusqu'à" "rond points rond")
  ("tenseur jusqu'à" "tenseur points tenseur")

  ("similaire à" "similaire")
  ("est similaire à" "similaire")
  ("équivalent" "similaire")
  ("équivalent à" "similaire")
  ("est équivalent" "similaire")
  ("est équivalent à" "similaire")
  ("asymptotique à" "asymptotique")
  ("est asymptotique à" "asymptotique")
  ("isomorphe à" "isomorphe")
  ("est isomorphe à" "isomorphe")

  ("négligeable devant" "négligeable")
  ("négligeable par rapport à" "négligeable")
  ("strictement dominé par" "négligeable")
  ("est strictement dominé par" "négligeable")
  ("dominé par" "dominé")
  ("est dominé par" "dominé")

  ("défini comme" "défini par")
  ("est défini par" "défini par")
  ("factorielle" "factoriel")

  ("la somme" "somme")
  ("le produit" "produit")
  ("l'intégrale" "intégrale")
  ("l'infini" "infini")

  ("grand chapeau" "large chapeau")
  ("grand tilde" "large tilde")
  ("grand barre" "large barre")
  ("en dessous" "dessous")
  )
