use ui;
use layout;
use core:geometry;
use progvis:net;
use core:io;

dialog ProblemsDlg {

	private Painter painter;

	public Action? action;

	init(Client client, progvis:Settings settings) {
		Callback cb(client, settings);

		ProblemData data;
		data.add("New problems", UnsolvedView(client, cb, false));
		data.add("Improved", UnsolvedView(client, cb, true));
		data.add("Solved", SolvedView(client, cb));
		data.add("My problems", OwnView(client, cb));

		init("Problems", Size(600, 500)) {
			painter(data);
		}

		cb.target = this;

		painter(painter);
	}

	Bool onClick(Bool down, Point pt, MouseButton button) : override {
		if (down & button == MouseButton:left) {
			painter.onClick(pt);
		}
		true;
	}

	Bool onMouseVScroll(Point at, Int delta) : override {
		painter.onVScroll(at, delta);
		true;
	}

	// Called from 'Callback' to indicate that we should close ourselves.
	void done(Action action) {
		this.action = action;
		close(1);
	}
}


/**
 * Callback for the dialog.
 */
class Callback on Ui {
	private Client client;
	private progvis:Settings settings;
	ProblemsDlg? target;

	init(Client client, progvis:Settings settings) {
		init {
			client = client;
		}
	}

	// Open a particular problem.
	void solveProblem(Int problemId) {
		unless (target)
			return;

		var d = details(problemId);
		var file = saveSource(d);
		if (d.allowSolution) {
			target.done(SolveProblem(d.id, d.title, file));
		} else {
			target.done(ShowProblem(d.id, d.title, file));
		}
	}

	// Show a solution to a problem.
	void showSolution(Int problemId, Str solution) {
		unless (target)
			return;

		var d = details(problemId);
		var file = saveSource(d);
		target.done(ShowSolution(d.id, d.title, solution, file));
	}

	// Get the problem we are to show from the server.
	private DetailsResponse details(Int id) {
		if (x = client.query(DetailsRequest(id)) as DetailsResponse)
			x;
		else
			throw InternalError("Incorrect response received.");
	}

	// Spill the source code to disk and receive a suitable Url for it.
	private Url saveSource(DetailsResponse response) {
		Url file = settings.downloadFile("${response.id.toS,f08}.${response.language}");
		Utf8Output out(file.write());
		out.autoFlush = false;
		out.write(response.source);
		out.flush();
		out.close();
		return file;
	}
}


/**
 * Action to take when the ProblemsDlg is closed.
 *
 * The base class contains basic information about the base-problem to display.
 */
class Action on Ui {
	// Which problem to solve.
	Int problemId;

	// Problem title.
	Str title;

	// Source code.
	Url source;

	init(Int id, Str title, Url source) {
		init {
			problemId = id;
			title = title;
			source = source;
		}
	}
}


/**
 * Show a problem, possibly trying to solve it as well.
 */
class SolveProblem extends Action {
	init(Int id, Str title, Url source) {
		init(id, title, source) {}
	}
}

/**
 * Show a problem that can not be solved for some reason.
 */
class ShowProblem extends Action {
	init(Int id, Str title, Url source) {
		init(id, title, source) {}
	}
}

/**
 * Show a solution to a problem.
 */
class ShowSolution extends Action {
	// Solution to show.
	Str solution;

	init(Int id, Str title, Str solution, Url source) {
		init(id, title, source) {
			solution = solution;
		}
	}
}
