/*
 * Strictly-ordered, singly-linked list to represent disjoint node ranges
 * of the type 'a' (single node) or 'a-b' (range, with a < b).
 *
 * For example, '1,7-8,20,33-29'
 *
 * Copyright (c) 2009-2011 Centro Svizzero di Calcolo Scientifico (CSCS)
 * Licensed under the GPLv2.
 */
#include "basil_alps.h"
#define CRAY_MAX_DIGITS	5	/* nid%05d format */

/* internal constructor */
static struct nodespec *ns_new(uint32_t start, uint32_t end)
{
	struct nodespec *new = calloc(1, sizeof(*new));

	if (new) {
		new->start = start;
		new->end   = end;
	}
	return new;
}

/** Return true if @candidate is in one of the ranges starting at @head */
bool ns_in_range(struct nodespec *head, uint32_t candidate)
{
	struct nodespec *cur;

	for (cur = head; cur; cur = cur->next)
		if (cur->start <= candidate && candidate <= cur->end)
			return true;
	return false;
}

/**
 * ns_add_range  -  Insert/merge new range into existing nodespec list.
 * @head:       head of the ordered list
 * @new_start:  start value of node range to insert
 * @new_end:    end value of node range to insert
 *
 * Maintains @head as duplicate-free list, ordered in ascending order of
 * node-specifier intervals, with a gap of at least 2 between adjacent entries.
 * Returns 0 if ok, -1 on failure.
 */
int ns_add_range(struct nodespec **head, uint32_t new_start, uint32_t new_end)
{
	struct nodespec *cur = *head, *next;

	assert(new_start <= new_end);

	if (cur == NULL || new_end + 1 < cur->start) {
		*head = ns_new(new_start, new_end);
		if (*head == NULL)
			return -1;
		(*head)->next = cur;
		return 0;
	}

	for (next = cur->next;
	     new_start > cur->end + 1;
	     cur = next, next = cur->next)
		if (next == NULL || new_end + 1 < next->start) {
			next = ns_new(new_start, new_end);
			if (next == NULL)
				return -1;
			next->next = cur->next;
			cur->next  = next;
			return 0;
		}

	/* new_start <= cur->end + 1 */
	if (new_start < cur->start)
		cur->start = new_start;

	if (new_end <= cur->end)
		return 0;
	cur->end = new_end;

	while ((next = cur->next) && next->start <= new_end + 1) {
		if (next->end > new_end)
			cur->end = next->end;
		cur->next = next->next;
		free(next);
	}
	/* next == NULL || next->start > new_end + 1 */

	return 0;
}

/** Add a single node (1-element range) */
int ns_add_node(struct nodespec **head, uint32_t node_id)
{
	return ns_add_range(head, node_id, node_id);
}

/**
 * ns_mergelists  -  Merge two nodespec lists
 * @to_head:   destination to merge into
 * @from_head: source to merge from (will be destroyed in the process)
 */
int ns_mergelists(struct nodespec **to_head, struct nodespec *from_list)
{
	struct nodespec *cur;
	int rc = 0;

	assert(to_head != NULL);

	for (cur = from_list; cur && rc == 0; cur = cur->next)
		rc = ns_add_range(to_head, cur->start, cur->end);

	free_nodespec(from_list);
	return rc;
}

/**
 * parse_nodelist  -  extract nodelist from string
 *
 * This accepts a more liberal nodespec syntax than the one used for output.
 * Input is accepted if it matches ^X(,X)*$, where X is defined as
 *
 *            X  =  [\s,]*[a-zA-Z]*\d+(-[a-zA-Z]*\d+)?[\s,]*
 *
 * Which means that:
 * - ranges a-b can be in any order, i.e. a <= b and a > b are both ok;
 * - alphabetical prefixes, such as 'nid0...' are ignored;
 * - leading zeroes are ignored;
 * - only positive decimal numbers are accepted, i.e. '0x' is not supported,
 *   and a leading '0' does not imply interpretation as octal number;
 * - leading/trailing whitespace, and whitespace around the ',' are ignored;
 * - leading/trailing ',' are ignored, as are repeated instances of ','
 *   (note that multiple X'es must be separated by at least one ',').
 */
struct nodespec *parse_nodelist(const char *nl)
{
	struct nodespec *head = NULL;
	uint32_t start, end;
	int rc = 0;

	if (nl == NULL)
		return NULL;

	do {
		while (isspace(*nl) || *nl == ',')
			nl++;

		while (isalpha(*nl))
			nl++;

		if (!isdigit(*nl))
			break;

		for (start = 0; isdigit(*nl); nl++)
			start = (*nl - '0') + 10 * start;

		if (*nl != '-') {
			end = start;
		} else {
			do
				++nl;
			while (isalpha(*nl));

			if (!isdigit(*nl))
				break;

			for (end = 0; isdigit(*nl); nl++)
				end = (*nl - '0') + 10 * end;
		}

		while (isspace(*nl))
			nl++;

		if (*nl != '\0' && *nl != ',')
			break;
		else if (start > end)
			rc = ns_add_range(&head, end, start);
		else
			rc = ns_add_range(&head, start, end);
	} while (rc == 0);

	if (rc || *nl) {
		free_nodespec(head);
		head = NULL;
	}
	return head;
}

/**
 * parse_node_array  -  Parse an array of nodespecs
 * @argnc: number of elements in @argns
 * @argns: array of nodespecs to parse
 */
struct nodespec *parse_node_array(int argnc, char *argns[])
{
	struct nodespec *cur, *head = NULL;
	int i;

	for (i = 0; i < argnc; i++) {
		cur = parse_nodelist(argns[i]);
		if (cur == NULL || ns_mergelists(&head, cur) != 0) {
			free_nodespec(head);
			return NULL;
		}
	}
	return head;
}

/**
 * compress_nodespecs  -  Combine multiple nodespecs into compressed string
 * @argnc: number of elements in @argns
 * @argns: array of nodespecs to parse
 */
char *compress_nodespecs(int argnc, char *argns[])
{
	struct nodespec *head = parse_node_array(argnc, argns);

	return head ? ns_to_string(head) : NULL;
}

/* count the number of nodes starting at @head */
int ns_count_nodes(const struct nodespec *head)
{
	const struct nodespec *cur;
	uint32_t node_count = 0;

	for (cur = head; cur; cur = cur->next)
		node_count += cur->end - cur->start + 1;

	return node_count;
}

/*
 * 	Pretty-printing routines and helpers
 */
static int num_digits(uint32_t val)
{
	int ndigits = 1;

	while (val /= 10)
		ndigits++;
	return ndigits;
}

/* Return the maximum number of digits needed to represent a node in @ns */
static int max_num_digits(const struct nodespec *ns)
{
	const struct nodespec *cur;
	int max_digits = 0;

	for (cur = ns; cur; cur = cur->next) {
		int n = num_digits(cur->end);

		if (n > max_digits)
			max_digits = n;
	}
	return max_digits;
}

/* Return minimum common zero-padding prefix for @ns */
static int min_common_zeropad(const struct nodespec *ns)
{
	int m = max_num_digits(ns);

	return m > CRAY_MAX_DIGITS ? 0 : CRAY_MAX_DIGITS - m;
}

/* print in compressed pdsh-like format */
void ns_print_pdsh_format(const struct nodespec *head, FILE *fp)
{
	const struct nodespec *cur;
	bool bracket_needed = head && (head->next || head->end > head->start);
	int zeropad = min_common_zeropad(head);

	if (head)
		fprintf(fp, "nid%0*u%s", zeropad, 0, bracket_needed ? "[" : "");

	for (cur = head; cur; cur = cur->next) {
		if (cur != head)
			fprintf(fp, ",");
		fprintf(fp, "%0*u", CRAY_MAX_DIGITS - zeropad, cur->start);
		if (cur->start != cur->end)
			fprintf(fp, "-%0*u",
				CRAY_MAX_DIGITS - zeropad, cur->end);
	}
	if (bracket_needed)
		fprintf(fp, "]");
}

/**
 * ns_ranged_string - Write compressed node specification to buffer.
 * @head:   start of nodespec list
 * @buf:    buffer to write to
 * @buflen: size of @buf
 * Returns number of characters written if successful, -1 on overflow.
 */
static ssize_t ns_ranged_string(const struct nodespec *head,
				char *buf, size_t buflen)
{
	const struct nodespec *cur;
	ssize_t n, len = 0;

	for (cur = head; cur; cur = cur->next) {
		if (cur != head) {
			n = snprintf(buf + len, buflen - len, ",");
			if (n < 0 || (len += n) >= buflen)
				return -1;
		}

		n = snprintf(buf + len, buflen - len, "%u", cur->start);
		if (n < 0 || (len += n) >= buflen)
			return -1;

		if (cur->start != cur->end) {
			n = snprintf(buf + len, buflen - len, "-%u", cur->end);
			if (n < 0 || (len += n) >= buflen)
				return -1;
		}
	}
	return len;
}

/* Compress @head into nodestring. Result must be free()d. */
char *ns_to_string(const struct nodespec *head)
{
	char *buf = NULL;
	size_t size = ns_count_nodes(head);

	if (size) {
		/* Over-estimation: using all digits, plus either '-' or '\0' */
		size *= CRAY_MAX_DIGITS + 1;

		buf = malloc(size);
		if (buf == NULL)
			fatal("can not allocate %d", (int)size);

		if (ns_ranged_string(head, buf, size) < 0)
			fatal("can not expand nodelist expression");
	}
	return buf;
}
