/*
 * ExtensionFileFilter.java - 0.9.0     01/07/2001 - 12:35:59
 *
 * Copyright (C) 2001,,2003 2002 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */
 
 
package net.sourceforge.groboutils.util.io.v1;
 
import java.io.IOException;
import java.io.File;
import java.io.FilenameFilter;



/**
 * Allows files with the given extention(s) to be accepted. You can also
 * specify whether directories are allowed or not. This filter is
 * case insensitive or sensitive, depending on the settings.  By default,
 * the class is case-insensitive.
 * <P>
 * The extension strings passed in are the end-of-name Strings, meaning
 * that each file must match at least one given string at the end. So,
 * if you want to match all "DOC" files, pass in ".doc" to match.
 * <P>
 * By default, directories are allowed, independent of their names. If
 * directories are not allowed, then the directory names must match the
 * extension list.
 *
 * @author   Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @since    January 7, 2001
 * @version  $Date: 2003/02/10 22:52:45 $
 */
public class ExtensionFilenameFilter implements FilenameFilter
{
    //--------------------------------------------------------------
    // Private fields
    
    // this list of extensions *must* be all lower-case.
    private String extensions[] = new String[0];
    private boolean allowDirs = true;
    private boolean caseInsensitive = true;
    
    
    //--------------------------------------------------------------
    // Constructors
    
    /**
     * Default Constructor.
     */
    public ExtensionFilenameFilter()
    {
        // do nothing
    }
    
    /**
     * Specify a single "end string" to match.
     */
    public ExtensionFilenameFilter( String extension )
    {
        addExtension( extension );
    }
    
    /**
     * <P>
     * Slow, but it works.
     */
    public ExtensionFilenameFilter( String exts[] )
    {
        if (exts == null)
        {
            throw new IllegalArgumentException( "no null args" );
        }
        for (int i = exts.length; --i >= 0;)
        {
            addExtension( exts[i] );
        }
    }
    
    
    /**
     * 
     */
    public ExtensionFilenameFilter( boolean caseInsensitive )
    {
        this.caseInsensitive = caseInsensitive;
    }
     
    
    /**
     * 
     */
    public ExtensionFilenameFilter( String extension, boolean caseInsensitive )
    {
        // must set the case first
        this.caseInsensitive = caseInsensitive;
        addExtension( extension );
    }

    
    /**
     * 
     */
    public ExtensionFilenameFilter( String exts[], boolean caseInsensitive )
    {
        if (exts == null)
        {
            throw new IllegalArgumentException( "no null args" );
        }

        // must set the case first
        this.caseInsensitive = caseInsensitive;
        for (int i = exts.length; --i >= 0;)
        {
            addExtension( exts[i] );
        }
    }

    
    //--------------------------------------------------------------
    // Public methods
    
    
    /**
     * Adds the given extension to the internal list.
     */
    public void addExtension( String ext )
    {
        if (ext == null)
        {
            throw new IllegalArgumentException( "no null args" );
        }
        int len = this.extensions.length;
        String temp[] = new String[ len+1 ];
        System.arraycopy( this.extensions, 0, temp, 0, len );
        if (isCaseInsensitive())
        {
            ext = ext.toLowerCase();
        }
        temp[ len ] = ext;
        this.extensions = temp;
    }
    
    
    /**
     * @return whether the filter allows directories to pass.
     */
    public boolean allowsDirectories()
    {
        return this.allowDirs;
    }
    
    
    /**
     * @param on <tt>true</tt> if filter allows directories to pass.
     */
    public void allowsDirectories( boolean on )
    {
        this.allowDirs = on;
    }
    
    
    /**
     * @return if the extension is case insensitive.
     */
    public boolean isCaseInsensitive()
    {
        return this.caseInsensitive;
    }
    
    
    /**
     * Accepts some files.
     */
    public boolean accept( File dir, String name )
    {
        if (name == null)
        {
            throw new IllegalArgumentException( "no null args" );
        }
        return matches( name, new File( dir, name ) );
    }
    

    
    //--------------------------------------------------------------
    // Protected methods
    
    
    /**
     * Tests if the given file matches the list of extensions.
     * 
     * @param filename the name of the file.
     * @param file the File object version of the file.
     */
    protected boolean matches( String filename, File file )
    {
        if (filename == null || file == null)
        {
            throw new IllegalArgumentException( "no null args" );
        }
        if (allowsDirectories() && file.isDirectory())
        {
            return true;
        }
        if (isCaseInsensitive())
        {
            filename = filename.toLowerCase();
        }
        
        for (int i = this.extensions.length; --i >= 0;)
        {
            if (filename.endsWith( this.extensions[i] ))
            {
                return true;
            }
        }
        return false;
    }
    
}
