/*	Copyright (C) 2018-2024 Martin Guy <martinwguy@gmail.com>
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation, either version 3 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* convert.c: Functions to convert one kind of value to another and
 * compute useful values from the global data. */

#include "spettro.h"
#include "convert.h"

#include "a_file.h"		/* for audio_file->sample_rate */
#include "ui.h"

#include <string.h>		/* for memmove(), strlen(), strchr() */
#include <ctype.h>		/* for toupper() */

/*
 * Vertical position (frequency domain) conversion functions
 *
 * The bottom pixel row (min_y) should be centered on the minimum frequency,
 * min_freq, and the top pixel row (max_y) on the maximum frequency, max_freq.
 */

/* Take "A0" or whatever and return the frequency it represents.
 * The standard form "C5#" is also recognized and synonym "C5+" for sharps
 * as well as "A4b" or "A4-" for flats.
 * Returns 0.0 if the note name is not recognized.
 */
freq_t
note_name_to_freq(const char *note)
{
    /* It goes A1 B1 C2 D2 E2 F2 G2 A2 B2 C3, so make An correct
     * and compensate for C-Gn being an octave lower */
    static int semitones[7] = { 0, 2, -9, -7, -5, -4, -2 }; /* A-G */
    bool sharp = note[0] && note[1] && (note[2] == '#' || note[2] == '+');
    bool flat  = note[0] && note[1] && (note[2] == 'b' || note[2] == '-');

    if (toupper(note[0]) >= 'A' && toupper(note[0]) <= 'G' &&
	(note[1] >= '0' && note[1] <= '9') &&
	(note[2] == '\0' || ((sharp || flat) && note[3] == '\0')))
	return (A4_FREQUENCY / 16.0) /* A0 */
		* pow(2.0, note[1] - '0') 
		* pow(2.0, (1/12.0) *
		           (semitones[toupper(note[0]) - 'A'] + sharp - flat));
    else
        return 0.0;
}

/* Convert a note number of the piano keyboard to the frequency it represents.
 * It's the note of an 88-note piano: 0 = Bottom A, 87 = top C
 */
freq_t
note_number_to_freq(const int n)
{
    static freq_t cache[88];	/* Init to 0.0 */
    if (cache[n] == 0.0)
	cache[n] = (A4_FREQUENCY / 16.0) /* A0 */
		   * pow(2.0, (1/12.0) * n);
    return cache[n];
}

/* Convert a frequency to a string for the frequency axis.
 *
 * Show up to five digits (up to 99999 or down to .0001) before going
 * to scientific notation, and show .0001, not 0.0001.
 * Apart from showing the normal audio frequency range with normal numbers,
 * this also means that the highest and lowest displayed
 * frequencies are wider than those nearer 1.0, which makes calculation
 * of the width of the frequency axis easier.
 */
char *
freq_to_string(freq_t freq)
{
    static char s[16];	/* [6] is probably enough */
    char *spacep;

    /* Left-align the number in the string */
    sprintf(s, "%-5.5g", freq);
    /* Turn 0.0001 into .0001 */
    if (s[0] == '0' && s[1] == '.') memmove(s, s+1, strlen(s));
    /* Remove trailing spaces */
    if ((spacep = strchr(s, ' ')) != NULL) *spacep = '\0';
    return s;
}

/*
 * Horizontal position (time domain) conversion functions
 *
 * We divide time into steps, one for each pixel column, starting from the
 * start of the piece of audio, with column 0 of the piece representing
 * what happens from 0.0 to 1/ppsec seconds.
 * The audio for the FFT for a given column should therefore be centered
 * on 
 *
 * disp_time, instead, is the exact time that we should be displaying at the
 * green line, in such a way that that exact time falls within the time
 * covered by that pixel column.
 */

/* Convert a time in seconds to the screen column in the whole piece that
 * contains this moment. */
int
time_to_piece_column(secs_t t)
{
    return (int) floor((t + secpp / 2) / secpp);
}

int
time_to_screen_column(secs_t t)
{
    return time_to_piece_column(t - disp_time) + disp_offset;
}

int
frames_to_piece_column(frames_t f)
{
    return time_to_piece_column(frames_to_secs(f));
}

int
frames_to_screen_column(frames_t f)
{
    return time_to_screen_column(frames_to_secs(f));
}

/* What time does the left edge of this screen column represent? */
secs_t
screen_column_to_start_time(int col)
{
    return disp_time + (col - disp_offset) * secpp;
}

/* What time, in frames, does the left edge of this screen column represent? */
frames_t
screen_column_to_start_frames(int col)
{
    return (frames_t) trunc(screen_column_to_start_time(col) * sr + (1/sr)/2);
}

/* What time in seconds does the start of this sample fall at? */
secs_t
frames_to_secs(frames_t frames)
{
    return frames / sr;
}

/* Which sample does this time in seconds fall in?
 * Each sample covers frames/sr <= secs < (frame+1)/sr
 */
frames_t
secs_to_frames(secs_t secs)
{
    return (frames_t) floor(secs * sr);
}

/*
 *	Choose a good FFT size for the given FFT frequency
 */

/* Helper functions */
static bool is_good_speclen(frames_t n);
static bool is_2357(frames_t n);

frames_t
fft_freq_to_speclen(freq_t fft_freq, freq_t sample_rate)
{
    frames_t speclen = (sample_rate / fft_freq + 1) / 2;
    frames_t d; /* difference between ideal speclen and preferred speclen */

    /* Find the nearest fast value for the FFT size. */

    for (d = 0 ; /* Will terminate */ ; d++) {
	/* Logarithmically, the integer above is closer than
	 * the integer below, so prefer it to the one below.
	 */
	if (is_good_speclen(speclen + d)) {
	    speclen += d;
	    break;
	}
	if (is_good_speclen(speclen - d)) {
	    speclen -= d;
	    break;
	}
    }

    return speclen;
}

/*
 * Helper function: is N a "fast" value for the FFT size?
 *
 * We use fftw_plan_r2r_1d() for which the documentation
 * http://fftw.org/fftw3_doc/Real_002dto_002dReal-Transforms.html says:
 *
 * "FFTW is generally best at handling sizes of the form
 *      2^a 3^b 5^c 7^d 11^e 13^f
 * where e+f is either 0 or 1, and the other exponents are arbitrary."
 *
 * Our FFT size is 2*speclen, but that doesn't affect these calculations
 * as 2 is an allowed factor and an odd fftsize may or may not work with
 * the "half complex" format conversion in calc_magnitudes().
 */

static bool
is_good_speclen (frames_t n)
{
    /* It wants n, 11*n, 13*n but not (11*13*n)
    ** where n only has as factors 2, 3, 5 and 7
     */
    if (n % (11 * 13) == 0) return 0; /* No good */

    return is_2357(n) || ((n % 11 == 0) && is_2357(n / 11))
		      || ((n % 13 == 0) && is_2357(n / 13));
}

/* Helper function: does N have only 2, 3, 5 and 7 as its factors? */
static bool
is_2357(frames_t n)
{
    /* Eliminate all factors of 2, 3, 5 and 7 and see if 1 remains */
    while (n % 2 == 0) n /= 2;
    while (n % 3 == 0) n /= 3;
    while (n % 5 == 0) n /= 5;
    while (n % 7 == 0) n /= 7;
    return (n == 1);
}

/* Convert time in seconds to a string like 1:30.45 */
char *
seconds_to_string(secs_t secs)
{
    unsigned h,m,s,f;	/* Hours, minutes, seconds and hundredths of a sec */
    unsigned isecs;	/* Number of whole seconds */
    static char string[sizeof("-HH:MM:SS.ff")]; /* Up to 99 hours */
    char *str = string;

    if (secs < 0.0) {
	/* Cannot happen (unless they set a bar line before time=0?)
	 * Do the right thing anyway. */
	string[0] = '-';
	str++;
	secs = -secs;
    }

    /* round to the nearest hundredth of a second */
    secs = round(secs * 100) / 100;
    isecs = (unsigned) trunc(secs);
    f = round((secs - (secs_t)isecs) * 100);
    s = isecs % 60;
    m = (isecs/60) % 60;
    h = isecs/60/60;

    {
	/* Avoid gcc compiler warning about possible sprintf buffer overflow.
	 * Sorry if your audio file is more than 255 hours long! */
	unsigned char ch = h;

	if (h > 0) sprintf(str, "%d:%02d:%02d.%02d", ch, m, s, f);
	else if (m > 0) sprintf(str, "%d:%02d.%02d", m, s, f);
	else sprintf(str, "%d.%02d", s, f);
    }

    return string;
}

/* Convert time in frames to a string like 1:30.45 */
char *
frames_to_string(frames_t frames)
{
    return seconds_to_string(frames / sr);
}

/* Convert a time string to a time in seconds.
 * The time may be any number of seconds (and maybe a dot and decimal places)
 * or minutes:SS[.dp] or hours:MM:SS[.dp]
 *
 * If the string argument is not parsable, we return a negative value.
 *
 * Other oddities forced by the use of sscanf():
 * Minutes with hours, and seconds with minutes, can also be a single digit.
 * Because %u and %f always discard initial white space, they can put spaces
 * after a colon or before the decimal point.
 * Seconds before a decimal point are %u instead of %2u because otherwise
 * "1:400.5" would be accepted as 1:40.5
 * "1:00:" is accepted as "1:00".
 */
secs_t
string_to_seconds(char *string)
{
    unsigned h, m, s;	/* Hours, minutes and seconds */
    secs_t frac = 0.0;	/* decimal places, 0 <= frac < 1.0 */
    secs_t secs;	/* Result for just the seconds */
    int n;		/* Number of chars were consumed by the match,
			 * used to detect trailing garbage */

    if (sscanf(string, "%2u:%2u:%u%lf%n", &h, &m, &s, &frac, &n) == 4) { }
    else
    if (sscanf(string, "%2u:%2u:%u%n", &h, &m, &s, &n) == 3) { }
    else
    if (sscanf(string, "%2u:%u%lf%n", &m, &s, &frac, &n) == 3)
	h = 0;	/* May have been set by a previous partial match */
    else
    if (sscanf(string, "%2u:%2u%n", &m, &s, &n) == 2)
	h = 0;
    else
    if (sscanf(string, "%lf%n", &secs, &n) == 1 &&
	    secs >= 0.0 && DELTA_LT(secs, (secs_t)(99*60*60 + 59*60 + 60)) &&
	    string[n] == '\0')
	return secs;
    else
	return -1.0;

    /* Handle all formats except a bare number of seconds */

    /* Range checks. Hours are limited to 2 digits when printed. */
    if (s > 59 || m > 59 || h > 99 || frac < 0.0 || frac >= 1.0 ||
	    string[n] != '\0')
	return -1.0;

    return h*60*60 + m*60 + s + frac;
}
