/*	Copyright (C) 2018-2024 Martin Guy <martinwguy@gmail.com>
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation, either version 3 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * a_cache.c - Stuff to keep any audio we might want in a memory buffer
 *
 * Previously, when an audio fragment was required, for the audio player or
 * for the FFT calculation threads, we would read that fragment of audio from
 * the audio file, with a lock to prevent multiple threads interfering with
 * each other (libmpg123 is thread safe, but we need to seek then read).
 *
 * This resulted in the calc threads and the audio-playing thread locking each
 * other out so much that spettro never used multiple CPUs to full advantage
 * and, with big FFT sizes, the calc threads would lock the audio player out.
 *
 * Instead, now, we always keep any audio that anyone might want to read
 * in a memory buffer, pre-emptively decoded whenever the playing position
 * changes. The audio that anyone might need is the displayed audio plus the
 * lookahead, plus half the FFT window size to the left and to the right.
 *
 * The cached audio needs to be refreshed whenever the displayed audio moves,
 * i.e. on user-interface pans and when the display scrolls while playing,
 * and when the time-zoom changes.
 */

#include "spettro.h"
#include "a_cache.h"

#include "calc.h"	/* for LOOKAHEAD */
#include "convert.h"	/* for frames_to_string() */
#include "lock.h"
#include "ui.h"

/* We keep the audio cache as a list of blocks of audio cache_step frames big.
 * When playing, we usually add new blocks to the end of the list and 
 * remove them from the start, so we keep a last-block pointer too.
 */
typedef struct ac_block {
    frames_t acb_start;
    frames_t acb_len;
    short *acb_s;	/* 16-bit all channels for audio */
    float *acb_f;	/* 32-bit mono floats for FFT threads */
    struct ac_block *acb_next;
} ac_block_t;
static ac_block_t *ac_blocks = NULL;
static ac_block_t *ac_last_block = NULL;

static void shorts_to_mono_floats(float *floats, short *shorts,
				  frames_t frames, int nchannels);

/*
 * read_cached_audio(): Same interface as read_audio_file().
 * Returns 0 if we are at end-of-file or a negative number if the audio
 * cache is mispositioned (which "should" never happen, but can if playing
 * position changes by a page but screen hasn't scrolled yet, or if a calc
 * takes so long to get done that time has aleady moved on).
 *
 * If there's some error, return -1.
 */

frames_t
read_cached_audio(audio_file_t *af, char *data,
		  af_format_t format, int channels,
		  frames_t start, frames_t frames_to_read)
{
    frames_t frames_written = 0;
    size_t framesize;	/* == sample size * number of channels */
    ac_block_t *acbp;

    if (getenv("NO_CACHE"))
	return read_audio_file(af, data, format, channels, start, frames_to_read);

    switch (format) {
    case af_float: framesize = sizeof(float); break;
    case af_signed: framesize = sizeof(short) * channels; break;
    default: abort();
    }

    /* Stuff we can do without referring to the audio cache, so no need
     * to lock it yet
     */

    /* Deal with start < 0 and fill with silence */
    if (start < 0) {
	frames_t nframes = MIN(-start, frames_to_read);
	bzero(data, nframes * framesize);
	start += nframes;
	frames_to_read -= nframes;
	frames_written += nframes;
	if (frames_to_read == 0) return frames_written;
    }

    /* Reading from before the start of the cache or from empty cache?
     * Happens if you scroll vigorously forward and some poor FFT thread
     * tries to fetch audio data from where you were when it started.
     */
    if (ac_blocks != NULL && start < ac_blocks->acb_start) {
	return -1;	/* Will make the FFT thread quit */
    }

    /* Now we need the audio_cache variables and memory to stay where they
     * are while we think about them and copy stuff out of the cache
     */
    lock_audio_cache();

    /* Scan through the list of blocks in the cache and
     * copy whichever parts overlap the requested region */
    for (acbp = ac_blocks; acbp != NULL; acbp=acbp->acb_next) {
	frames_t copy_start, copy_len;

	/* If the end of the block is before the start of the region, seek on */
	if (acbp->acb_start + acbp->acb_len <= start) continue;

	/* Stop if we've gone past the requested region */
	if (acbp->acb_start >= start + frames_to_read) break;

	/* They overlap. Copy the common region */
	copy_start = MAX(acbp->acb_start, start);
	copy_len   = MIN((acbp->acb_start + acbp->acb_len) - start,
			 frames_to_read);
	switch (format) {
	case af_float:
	    memcpy(data + frames_written * framesize,
		   (char *)(acbp->acb_f + (copy_start - acbp->acb_start)),
		   copy_len * framesize);
	    break;
	case af_signed:
	    memcpy(data + frames_written * framesize,
		   (char *)(acbp->acb_s + (copy_start - acbp->acb_start) * channels),
		   copy_len * framesize);
	    break;
	}
	frames_written += copy_len;
	frames_to_read -= copy_len;
	start += copy_len;
    }

    unlock_audio_cache();

    /* Does it read past end-of-cache? If so, fill the end with silence */
    if (frames_to_read > 0) {
	bzero(data + frames_written * framesize, frames_to_read * framesize);
	frames_written += frames_to_read;
    }

    return(frames_written);
}

static void fill_block(ac_block_t *block);

/*
 * Make the cached portion of the audio reflect the current settings:
 * the displayed portion of the audio with lookahead plus
 * half the FFT window size at each end.
 *
 * This should only be called from the main thread.
 */
void
reposition_audio_cache()
{
    frames_t new_cache_start;/* Where the audio cache will start,
			      * in frames from start of audio file */
    secs_t new_cache_time;  /* How big will the new cache be, in seconds */
    frames_t new_cache_size;/* How big will it be, in sample frames */
    frames_t cache_step;    /* How many frames to step it by */
    frames_t fill_start;    /* When new and old caches overlap, this is the */
    int nchannels;	    /* Local copy */
    ac_block_t **acbpp;	    /* List-scanning pointer points to ac_blocks or to
    			     * the acb_next field of the block previous to the
			     * one we're interested in */

    if (getenv("NO_CACHE")) return;

    /* Align the cache and its size to a multiple of some step. */
    cache_step = 10240;

    nchannels = current_audio_file()->channels;

    new_cache_start = screen_column_to_start_frames(min_x)
		      - secs_to_frames((1.0/fft_freq)/2);
    new_cache_time = ((max_x + 1 - min_x) + LOOKAHEAD) * secpp + 1.0/fft_freq;
    new_cache_size = secs_to_frames(new_cache_time);

    /* Limit the cached area to the start of the audio file */
    if (new_cache_start < 0) {
	new_cache_size -= -new_cache_start;
	new_cache_start = 0;
	if (new_cache_size < 0) {
	    fprintf(stderr, "Moving all of the audio cache before the start of the audio!\n");
	    /* Impossible. Refuse to move it */
	    return;
	}
    }
    /* and to the end of the audio file */
    if (new_cache_start + new_cache_size > current_audio_file()->frames) {
	if (new_cache_start > current_audio_file()->frames) {
	    fprintf(stderr, "new_cache_start exceeds audio file length!\n");
	    /* Impossible. Refuse to move it */
	    return;
	}
	new_cache_size = current_audio_file()->frames - new_cache_start;
    }

    /* Step back by up to (cache_step - 1) */
    new_cache_start = new_cache_start
		      / cache_step * cache_step;
    /* and increase its size to have up to cache_step more than we need */
    new_cache_size = (new_cache_size + (cache_step + cache_step - 1))
		      / cache_step * cache_step;

    lock_audio_cache();

    /* Free any saved blocks that are before the new cached area */
    for (acbpp = &ac_blocks; *acbpp != NULL; /* reinit in loop */) {
	ac_block_t *block_to_free;
	/* If this block is within or past the new cache,
	 * we've freed all blocks previous to the new region. */
	if ((*acbpp)->acb_start + (*acbpp)->acb_len > new_cache_start)
	    break;
	/* We are freeing from the start of the block list */
	block_to_free = *acbpp;
	ac_blocks = (*acbpp)->acb_next;
	acbpp = &ac_blocks;
	free(block_to_free->acb_f);
	free(block_to_free->acb_s);
	free(block_to_free);
    }
    /* acbpp now points to ac_blocks or to the acb_next field of the first
     * block that coincides with or is after the start of the new cached area */

    /* Ensure all blocks of the new cache are in the block list */
    for (fill_start = new_cache_start;
	 fill_start < new_cache_start + new_cache_size;
	 fill_start += cache_step) {
	if ((*acbpp) == NULL) {
	    /* The cache is empty or we are at the end of the cache.
	     * Append the new one to block list. */
	    ac_block_t *new_block = Malloc(sizeof(ac_block_t));
	    new_block->acb_start = fill_start;
	    new_block->acb_len = cache_step;
	    new_block->acb_f = Malloc(cache_step * sizeof(float));
	    new_block->acb_s = Malloc(cache_step * sizeof(short) * nchannels);
	    fill_block(new_block);
	    new_block->acb_next = NULL;
	    *acbpp = new_block;
	    ac_last_block = new_block;
	} else {
	    ac_block_t *new_block;

	    /* Step past blocks that are before the one we want to fill */
	    while ((*acbpp) != NULL && (*acbpp)->acb_start < fill_start)
		acbpp = &((*acbpp)-> acb_next);

	    /* Is it already in the cache? */
	    if ((*acbpp) != NULL && (*acbpp)->acb_start == fill_start)
		/* Yes */
		continue;

	    /* No. Allocate and fill it */
	    new_block = Malloc(sizeof(ac_block_t));
	    new_block->acb_start = fill_start;
	    new_block->acb_len = cache_step;
	    new_block->acb_f = Malloc(cache_step * sizeof(float));
	    new_block->acb_s = Malloc(cache_step * sizeof(short) * nchannels);
	    fill_block(new_block);

	    /* If we're at the end of the list, append it there */
	    if ((*acbpp) == NULL) {
		*acbpp = new_block;
		new_block->acb_next = NULL;
		ac_last_block = new_block;
	    } else {
		/* Nope. Add the new block after the previous one */
		new_block->acb_next = *acbpp;
		*acbpp = new_block;
		if (new_block->acb_next == NULL) ac_last_block = new_block;
	    }
	}
    }
    /* We leave blocks that are later than the new cache size
     * as they cost us only memory, not speed. */

    unlock_audio_cache();
}

/* Fill a cache block from the audio file */
static void
fill_block(ac_block_t *block)
{
    int nchannels = current_audio_file()->channels;
    frames_t r;	/* Number of frame read */

    /* Read 16-bit nchannel shorts into the buffer
     * and convert those to mono floats */
    r = read_audio_file(current_audio_file(), (char *)block->acb_s, af_signed,
			nchannels, block->acb_start, block->acb_len);
    if (r < 0) {
	fprintf(stderr, "Read error on audio file for %ld frames at %s\n",
		block->acb_len, frames_to_string(block->acb_start));
	return;
    }
    if (r != block->acb_len) {
	fprintf(stderr, "Failed to fill the audio cache with %ld frames; got %ld.\n",
		block->acb_len, r);
    }
    shorts_to_mono_floats(block->acb_f, block->acb_s, r, nchannels);

    if (r < block->acb_len) {
	fprintf(stderr, "Filling misread area with %ld frames of silence\n",
		block->acb_len - r);
	bzero(block->acb_f + r,
	       (block->acb_len - r) * sizeof(float));
	bzero(block->acb_s + r * nchannels,
	       (block->acb_len - r) * sizeof(short) * nchannels);
    }
}

/* Convert 16-bit nchannel shorts to mono floats */
static void
shorts_to_mono_floats(float *floats, short *shorts, frames_t frames, int nchannels)
{
    float *fp; short *sp;
    frames_t todo;

    sp=shorts, fp = floats, todo = frames;
    switch (nchannels) {
    case 1:
	while (todo > 0) {
	    *fp++ = (float)(*sp++) / 32768.0;
	    todo--;
	}
	break;
    case 2:
	while (todo > 0) {
	    *fp++ = (float)((long)sp[0] + (long)sp[1]) / 65536.0;
	    sp += 2; todo--;
	}
	break;
    default:
	while (todo > 0) {
	    int i;
	    int total = 0;

	    for (i = 0; i < nchannels; i++) total += *sp++;
	    *fp++ = (float)total / (32768 * nchannels);
	    todo--;
	}
    }
}

/* Dump the audio cache as a WAV file */
void
dump_audio_cache()
{
#if !HAVE_LIBSNDFILE
    fprintf(stderr, "Dumping the audio cache requires libsndfile\n");
#else
    SF_INFO  sfinfo;
    SNDFILE *sf;
    const char filename[]="audio_cache.wav";
    int nchannels = current_audio_file()->channels;
    ac_block_t *acbp;

    if (getenv("NO_CACHE")) return;

    sfinfo.samplerate = sr;
    sfinfo.channels = nchannels;
    sfinfo.format = SF_FORMAT_WAV | SF_FORMAT_PCM_16;

    sf = sf_open(filename, SFM_WRITE, &sfinfo);
    if (sf == NULL) {
	fprintf(stderr, "Cannot create ");
	perror(filename);
	return;
    }

    for (acbp = ac_blocks; acbp != NULL; acbp=acbp->acb_next) {
	sf_count_t frames_written;
	frames_written = sf_writef_short(sf, acbp->acb_s, acbp->acb_len);
	if (frames_written != acbp->acb_len) {
	    fprintf(stderr, "Failed to write audio cache\n");
	}
    }

    fprintf(stderr, "Dumped the audio cache to %s\n", filename);
    sf_close(sf);
#endif
}

void
print_audio_cache_stats(void)
{
    frames_t cache_size;
    ac_block_t *acbp;		/* Loop pointer for scanning the list */
    ac_block_t *prev;		/* The previous block to *acbp */

    if (getenv("NO_CACHE")) {
	printf("No audio cache\n");
	return;
    }

    if (ac_blocks == NULL && ac_last_block == NULL) {
	printf("The audio cache is empty\n");
	return;
    }
    /* Consistency check */
    if (ac_blocks == NULL) { 
	printf("Internal error: ac_blocks is NULL\n");
	return;
    }
    if (ac_last_block == NULL) {
	printf("Internal error: ac_last_block is NULL\n");
	return;
    }

    cache_size = 0; prev = NULL;
    for (acbp = ac_blocks; acbp != NULL; acbp=acbp->acb_next) {
	/* Cache blocks should be consecutive in time */
	if (prev != NULL && acbp->acb_start != prev->acb_start + prev->acb_len){
	    printf("Internal error: Cache blocks are not consecutive: ");
	    printf("%ld(%ld) - %ld(%ld)\n", prev->acb_start, prev->acb_len,
					    acbp->acb_start, acbp->acb_len);
	}
	cache_size += acbp->acb_len;
	prev = acbp;
    }

    printf("audio cache runs from %s ",
	    frames_to_string(ac_blocks->acb_start));
    printf("to %s of %ld frames\n",
	    frames_to_string(ac_last_block->acb_start + ac_last_block->acb_len - 1),
	    cache_size);
}
