/* ----------------------------------------------------------------

ImageEnlarger  -  resize, especially magnify bitmaps in high quality
    Array3D.cpp: 2-dim. vector-array

Copyright (C) 2009 Mischa Lusteck

This program is free software;
you can redistribute it and/or modify it under the terms of the
GNU General Public License as published by the Free Software Foundation;
either version 3 of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY;
without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, see <http://www.gnu.org/licenses/>.

---------------------------------------------------------------------- */

#include <string>
#include <iostream>
#include <math.h>

#include "Array3D.h"
#include "Array.h"
#include "ArraysTemplateDefs.h"

template class BasicArray<Point>;  // explicit instantiation 

void  MyArray3D::Clamp01(void) {
   int x,y;
   int sizeX = SizeX(), sizeY = SizeY();
   for (y=0; y<sizeY; y++ )   {
      for (x=0; x<sizeX; x++ )   {
         Point p=Get(x,y);
         p.Clip();
         Set(x,y,p);
      }
   }
}

void MyArray3D::HiSharpen0(float f) {
   MyArray3D src(*this);
   int x,y;
   int sizeX = SizeX(), sizeY = SizeY();
   float invF = 1.0/f;
   
   for(y=1;y<sizeY-1;y++) {
      for(x=1;x<sizeX-1;x++) {
         float dd=0.0,ff;
         Point l,c;
         
         c = src.Get( x  , y );
         ff = c.Norm1();
         ff = 1.0 - 8.0*ff;
         if(ff<0.0)ff=0.0;
         ff*=ff;
         ff*=ff;
         ff*=ff;
         ff*=ff;
         Mul(x,y,1.0 + f*ff);
      }
   }
}
void MyArray3D::HiSharpen(float f) {
   MyArray3D src(*this);
   int x,y;
   int sizeX = SizeX(), sizeY = SizeY();
   float invF = 1.0/f;
   
   for(y=1;y<sizeY-1;y++) {
      for(x=1;x<sizeX-1;x++) {
         float dd=0.0,ff;
         Point l,c;
         c = src.Get( x  , y );
         l = src.Get( x  , y-1 ) - c; dd+=l.Norm1();
         l = src.Get( x-1, y   ) - c; dd+=l.Norm1();
         l = src.Get( x+1, y   ) - c; dd+=l.Norm1();
         l = src.Get( x  , y+1 ) - c; dd+=l.Norm1();
         dd*=2.0;
         l = src.Get( x-1, y-1 ) - c; dd+=l.Norm1();
         l = src.Get( x+1, y-1 ) - c; dd+=l.Norm1();
         l = src.Get( x-1, y+1 ) - c; dd+=l.Norm1();
         l = src.Get( x+1, y+1 ) - c; dd+=l.Norm1();
         dd*=(1.0/12.0);
         
         dd = 10.0*f*dd;
         if(dd>1.0)dd=1.0;
         dd = pow_f(dd,2.0);
         //dd = dd*dd*(3.0 - 2.0*dd);
         
         ff = c.Norm1();
         ff = 1.0/(10.0*ff+ 0.001) - 1.0/3.0001;
         Mul(x,y,1.0 + 0.2*dd*ff);
      }
   }
}

void MyArray3D::ReduceNoise0(void) {
   const float f = 0.015;  
   const float invF=1.0/f;

   MyArray3D src(*this);
   int x,y;
   int sizeX = SizeX(), sizeY = SizeY();

   for(y=1;y<sizeY-1;y++) {
      for(x=1;x<sizeX-1;x++) {
         Point l;
         float dd=0.0;
         
         // higher gradient -> reduce less noise
         l  = src.Get( x+1, y+1 ) - src.Get( x-1, y-1 );
         dd+=l.Norm1();
         l  = src.Get( x-1, y+1 ) - src.Get( x+1, y-1 );
         dd+=l.Norm1();
         l  = src.Get( x+1, y   ) - src.Get( x-1, y   );
         dd+=2.0*l.Norm1();
         l  = src.Get( x  , y+1 ) - src.Get( x  , y-1 );
         dd+=2.0*l.Norm1();
         dd*=(1.0/12.0);
         dd = 2.0*f - dd;
         if(dd<0.0)
            dd=0.0;
         else {
            dd *= (1.0/2.0) * invF;
            dd = dd*dd*(3.0 - 2.0*dd); 
         }

         l += src.Get( x+1, y   ) + src.Get( x  , y+1 );
         l*=2.0;
         l += src.Get( x-1, y-1 ) + src.Get( x+1, y-1 );
         l += src.Get( x-1, y+1 ) + src.Get( x+1, y+1 );
         l*=(1.0/12.0);
         l -= src.Get( x  , y   );

         l  = src.Get( x  , y-1 ) + src.Get( x-1, y   );
         l += src.Get( x+1, y   ) + src.Get( x  , y+1 );
         l*=2.0;
         l += src.Get( x-1, y-1 ) + src.Get( x+1, y-1 );
         l += src.Get( x-1, y+1 ) + src.Get( x+1, y+1 );
         l*=(1.0/12.0);
         l -= src.Get( x  , y   );
         float nn = l.Norm1();
         if(nn<f) {
            nn *= invF;
            nn = nn*nn*(3.0 - 2.0*nn);
            l *= nn*dd;
            l = src.Get( x  , y   ) + l;
            l.Clip();
            Set(x,y,l);
         }
      }
   }
}

void MyArray3D::ReduceNoise(float reduceF) {
   MyArray3D *hiF,*loF;
   int x,y;
   int sizeX = SizeX(), sizeY = SizeY();
   
   if(reduceF<=0.0)
      return;
   reduceF =1.0/reduceF;
   
   hiF = new MyArray3D(*this);
   loF = hiF->SplitLowFreq(1);
   
   for(y=0;y<sizeY;y++) {
      for(x=0;x<sizeX;x++) {
         Point p;
         float w,dd;

         p = hiF->Get(x,y);
         dd = p.Norm1()*5.0*reduceF;
         if(dd<1.0) {
            w = dd;
            w*=(2.0 - w);
            w*=(2.0 - w);
            w*=(2.0 - w);
            
            // create smooth ending
            if(w<0.2) {            
               w = w*5.0;       
               w = 0.1*w*w + 0.1;
            }
            w =  1.1*(w - 1.0) + 1.0;  
            
            p *= (1.0 - w);
            Sub(x,y,p);
         }
      }
   }
   delete hiF;
   delete loF;
}

MyArray *MyArray3D::NormArray(void) {
   int x,y;
   int sizeX = SizeX(), sizeY = SizeY();

   MyArray *a4 = new MyArray(sizeX,sizeY);

   for (y=0; y<sizeY; y++ )   {
      for (x=0; x<sizeX; x++ )   {
         float nn = Get(x,y).Norm1()*2.0;
         if(nn>1.0)nn=1.0;
         a4->Set(x,y,nn);
      }
   }
   return a4;
}

//
// Calculate array of directions
// it contains the directions of found edges
// direction is orthogonal to gradient, but without front-rear-information
// (more like projective coordinates)
// which is simply done by internally doubling the angles ( complex squaring of (x,y) )
// and thus identifying (x,y) and -(x,y)
// therefore orthogonal directions negate each other

DirArray *MyArray3D::Direction(void) {
   int x,y;
   int sizeX = SizeX(), sizeY = SizeY();
   Point2 gradR,gradG,gradB,dirTotal,p2;     // total direction is avg. of R,G,B dir.
   DirArray *result = new DirArray(sizeX,sizeY);

   for (y=1; y<sizeY-1; y++ )   {
      for (x=1; x<sizeX-1; x++ )   {
         Point dx = DX(x,y);
         Point dy = DY(x,y);
         gradR.x   = dy.x; gradR.y = -dx.x;
         dirTotal  = gradR.DoubleDir();
         gradG.x   = dy.y; gradG.y = -dx.y;
         dirTotal += gradG.DoubleDir();
         gradB.x   = dy.z; gradB.y = -dx.z;
         dirTotal += gradB.DoubleDir();
         dirTotal *= (1.0/3.0);
         result->Set(x,y,dirTotal);
      }
   }

   return result;
}

void MyArray3D::SmoothenWithDir(DirArray *dir) {
   int x,y,ax,ay,a;
   int sizeX = SizeX(), sizeY = SizeY();
   
   MyArray3D src(*this);

   for (y=1; y<sizeY-1; y++ )   {
      for (x=1; x<sizeX-1; x++ )   {
         float w[9],ww,wTotal;
         
         a=0;
         wTotal = 0.0;
         for(ay=0;ay<3;ay++) {
            for(ax=0;ax<3;ax++) {
               Point2 dM = Point2(float(ax-1),float(ay-1));
               dM = dM.DoubleDir();
               ww = 1.0 + 20.0*dM*dir->Get(x-1+ax,y-1+ay);
               if(ww<0.000001)
                  ww=0.000001;
               if(ax==1)ww*=2.0;
               if(ay==1)ww*=2.0;
               w[a] = ww;
               wTotal += ww;
               a++;
            }
         }
         ww = 1.0/wTotal;
         for(a=0;a<9;a++)
            w[a]*=ww;
         
         a=0;
         Point pSmooth(0.0);
         for(ay=0;ay<3;ay++) {
            for(ax=0;ax<3;ax++) {
               pSmooth += w[a]*src.Get(x-1+ax,y-1+ay);
               a++;
            }
         }
         Set(x,y,pSmooth);
      }
   }
}

void MyArray3D::SmoothenWithDir0(void) {
   DirArray *dir;
   int x,y,ax,ay,a;
   int sizeX = SizeX(), sizeY = SizeY();
   
   dir = Direction();

   MyArray3D src(*this);

   for (y=1; y<sizeY-1; y++ )   {
      for (x=1; x<sizeX-1; x++ )   {
         float w[9],ww,wTotal;
         
         a=0;
         wTotal = 0.0;
         Point2 dM = dir->Get(x,y);
         for(ay=0;ay<3;ay++) {
            for(ax=0;ax<3;ax++) {
               ww = 1.0 + 40.0*dM*dir->Get(x-1+ax,y-1+ay);
               if(ww<0.000001)
                  ww=0.000001;
               if(ax==1)ww*=2.0;
               if(ay==1)ww*=2.0;
               w[a] = ww;
               wTotal += ww;
               a++;
            }
         }
         ww = 1.0/wTotal;
         for(a=0;a<9;a++)
            w[a]*=ww;
         
         a=0;
         Point pSmooth(0.0);
         for(ay=0;ay<3;ay++) {
            for(ax=0;ax<3;ax++) {
               pSmooth += w[a]*src.Get(x-1+ax,y-1+ay);
               a++;
            }
         }
         Set(x,y,pSmooth);
      }
   }
   
   delete dir;
}

void MyArray3D::AddWaves(float f) {
   int x,y;
   int sizeX = SizeX(), sizeY = SizeY();
   float fx,fy;
      
   for (y=0; y<sizeY; y++ )   {
      for (x=0; x<sizeX; x++ )   {
         Point p;
         fx = float(x)*f*0.1;
         fy = float(y)*f*0.1;
         p.x = 0.5*cos(  3.0*fx - 1.5*fy  + 12.3 ) + 0.5;
         p.y = 0.5*cos(  1.2*fx + 2.5*fy  +  4.5 ) + 0.5;
         p.z = 0.5*cos( -0.4*fx + 3.5*fy  + 34.2 ) + 0.5;
         p = 0.5*p + 0.5*Get(x,y);
         Set(x,y,p);
      }
   }
}

